"""
Demonstrates how to add a menu and/or a tool bar to a Traits UI View.

A menu bar is constructed as a hierarchy of MenuBar, Menu, Action and
Separator items:
    
    MenuBar(
        Menu(
            Menu(
                Action(...),
                Action(...),
                Separator(),
                Action(...)
                ...
                name = 'menu name'
            ),
            Menu(...),
            Action(...),
        Menu(
            Action(...),
            Action(...),
            ...
            name = 'menu name'
        )
    )
    
Similarly, a toolbar is constructed using ToolBar, Action and Separator items:
    
    ToolBar(
        Action(...),
        Action(...),
        Separator(),
        Action(...),
        ...
    )
    
Menus and tool bars can share common Action items. In the case of tool bar
actions, you can also specify an 'image' trait to define an image to display 
for the Action on the tool bar. This trait is ignored if the Action is used in 
a menu.

A menu bar is added to a View using the 'menubar' trait. A tool bar is added
using the 'toolbar' trait.

There are also a number of optional traits defined on an Action which can
make the Action behave differently at run-time base on the current state of
the application:
    
    - enabled_when: Controls when the action is enabled/disabled.
    - defined_when: Controls when the action is included/excluded from the menu
                    or tool bar.
    - visble_when:  Controls when the action is visble/hidden.
    - checked_when: Controls when a menu action is checked or not.
    
The handler for a given Action is a method with the same name as the 'action'
trait on an Action item. The method must be defined on the subclass of the
Traits UI Handler class that is associated with the Traits UI View the menu
or tool bar is contained in.
"""

from enthought.traits.api \
    import HasTraits, Str, Enum, List, Instance
    
from enthought.traits.ui.api \
    import View, Item, Label, Handler, TableEditor
    
from enthought.traits.ui.table_column \
    import ObjectColumn
    
from enthought.traits.ui.menu \
    import MenuBar, ToolBar, Menu, Action, NoButtons
    
from enthought.pyface.image_resource \
    import ImageResource

# Define the search path for the Action 'image' files
# (this is only needed because the demo file is started using 'execfile'):    
import os
search_path = os.path.join( os.getcwd(), 'images' )

# The Employee class:
class Employee ( HasTraits ):
    
    # The employee's first name:
    first_name = Str
    
    # The employee's last name:
    last_name = Str
    
    # The employee's department:
    department = Enum( 'Development', 'Research', 'Quality Assurance',
                       'Payroll', 'Human Resources' )
                       
    # The view for creating a new employee:
    view = View(
        Item( 'first_name' ),
        Item( 'last_name' ),
        Item( 'department' ),
        title   = 'Add a New Employee',
        width   = 0.2,
        buttons = [ 'OK', 'Cancel' ]
    )        
                       
# Defne the employee table editor:
table_editor = TableEditor(
    columns = [ ObjectColumn( name = 'first_name', width = 0.3 ),
                ObjectColumn( name = 'last_name',  width = 0.3 ),
                ObjectColumn( name = 'department', width = 0.4 ) ],
    selected     = 'selected',
    sortable     = False,
    deletable    = False,
    editable     = False,
    configurable = False
)  

# Define common actions shared by the menu bar and tool bar:

# 'Add a new employee' action:
add_action = Action( 
    name   = 'Add Employee...', 
    action = 'add_employee',
    # Normally you can leave the 'search_path' argument out, but not in this
    # case because the demo is started using 'execfile' which does not define
    # the module '__name__', so we specify an explicit search path instead:
    #image = ImageResource( 'add_employee' )
    image  = ImageResource( 'add_employee', search_path = [ search_path ] )
)
                     
# 'Delete an existing employee' action:
delete_action = Action(
    name   = 'Delete Employee',
    action = 'delete_employee',
    enabled_when = 'object.selected is not None',
    # Normally you can leave the 'search_path' argument out, but not in this
    # case because the demo is started using 'execfile' which does not define
    # the module '__name__', so we specify an explicit search path instead:
    #image = ImageResource( 'delete_employee' )
    image  = ImageResource( 'delete_employee', search_path = [ search_path ] )
)

# Define the application menu bar:
menubar = MenuBar(
    Menu( 
        Action( name   = 'Quit', 
                action = 'quit' ),
        name = 'File'
    ),
    Menu( add_action, delete_action, name = 'Edit' ),
    Menu( 
        Action( name   = 'About...',
                action = 'show_about' ),
        name = 'Help'
    )
)    
 
# Define the application tool bar:
toolbar = ToolBar( add_action, delete_action )

class HRDemoHandler ( Handler ):
    """ Define the handler class for the HRDemo view. This class defines all
        of the handler methods for the menu and tool bar actions."""
    
    def quit ( self, info ):
        """ Quit the application. """
        info.ui.dispose()
        
    def add_employee ( self, info ):
        """ Add a new employee. """
        new = Employee()
        ui  = new.edit_traits( kind = 'livemodal' )
        if ui.result:
            info.object.employees.append( new )
        
    def delete_employee ( self, info ):
        """ Delete an existing employee. """
        hr_demo = info.object
        hr_demo.employees.remove( hr_demo.selected )
        hr_demo.selected = None
        
    def show_about ( self, info ):
        """ Display the 'About' view. """
        self.edit_traits( view = 'about' )
        
    # The 'About' view:
    about = View( 
        Label( "This is the 'About' view." ),
        title   = 'About Human Resources Administration',
        width   = 0.25,
        buttons = [ 'OK' ]
    )
                       
# The menu and tool bar demo class:
class HRDemo ( HasTraits ): 

    # The list of employees:
    employees = List( Employee )
    
    # The currently selected employee:
    selected = Instance( Employee )
    
    # The application view:
    view = View(
        Item( 'employees',
              show_label = False,
              editor     = table_editor ),
        title     = 'Human Resources Administration',
        width     = 0.3,
        height    = 0.3,
        resizable = True,
        buttons   = NoButtons,
        menubar   = menubar,
        toolbar   = toolbar,
        handler   = HRDemoHandler
    )

# Create some sample employees:
employees = [
    Employee( first_name = 'Mark',
              last_name  = 'Thompson', 
              department = 'Development' ),
    Employee( first_name = 'Millie',
              last_name  = 'Dickerson', 
              department = 'Research' ),
    Employee( first_name = 'Wilfred',
              last_name  = 'Jones', 
              department = 'Payroll' ),
    Employee( first_name = 'Julie',
              last_name  = 'Williams', 
              department = 'Quality Assurance' ),
    Employee( first_name = 'George',
              last_name  = 'Montgomery', 
              department = 'Human Resources' ),
]

# Hook for 'demo.py'
popup = HRDemo( employees = employees )

if __name__ == "__main__":
    popup.configure_traits()
